/**
 * \file: InputChannel.h
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: CarPlay
 *
 * \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
 *
 * \copyright (c) 2013-2014 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include <HIDUtils.h>
#include <dipo_macros.h>
#include "AirPlayHeaders.h"
#include "Session.h"
#include "InputChannel.h"
#include "utils/Utils.h"

using namespace std;

namespace adit { namespace carplay
{

InputChannel::InputChannel(Session& session) : session(session)
{
    verbose = false;
}

InputChannel::~InputChannel()
{
    // remove all HID devices
    for (auto device : hidDevices)
    {
        auto ref = device.second;

        //De-register devices
        if (kNoErr !=  HIDDeregisterDevice(ref))
            LOG_WARN((dipo, "failed to de-register HID device"));

        // remove from HID browser
        HIDDeviceForget(&ref);
    }
    hidDevices.clear();

    // destroy all adapters
    inputAdapters.clear();

    LOGD_DEBUG((dipo, "InputChannel stopped"));
}

bool InputChannel::Initialize(IConfiguration* config)
{
    auto inputAdapterNames = config->GetItems("IInputAdapter");
    for (auto impl : inputAdapterNames)
    {
        unique_ptr<IInputAdapter> ptr(static_cast<IInputAdapter*>(
                Factory::Instance()->Create(impl)));
        if (ptr == nullptr)
        {
            LOG_WARN((dipo, "%s implementation %s not found!", "IInputAdapter", impl.c_str()));
            continue;
        }
        if (!ptr->Initialize(*config, *this, static_cast<SessionId>(&session)))
        {
            LOG_WARN((dipo, "Could not initialize %s!", impl.c_str()));
            continue;
        }
        inputAdapters.push_back(move(ptr));
    }

    if (inputAdapters.empty())
    {
        // this is clearly an error, but don't stop DiPO
        LOG_WARN((dipo, "no input adapters loaded!"));
    }

    string verboseConfig;
    if (config->TryGetItem("enable-verbose-logging", verboseConfig))
    {
        verbose = 0 != config->GetNumber("enable-verbose-logging", 0);
        if (verbose)
        {
            LOG_WARN((dipo, "input: verbose logging enabled, do not use in production!"));
        }
    }

    return true;
}

bool InputChannel::AttachInput(const HIDDevice& device)
{
    if (hidDevices.end() == hidDevices.find(device.UUID))
    {
        LOGD_DEBUG((dipo, "HID attach: %s %s", device.Name, device.UUID));
        auto properties = CFDictionaryCreateMutable(NULL, 0, &kCFTypeDictionaryKeyCallBacks,
                &kCFTypeDictionaryValueCallBacks);

        // set name
        CFDictionarySetCString(properties, kHIDDeviceProperty_Name, device.Name, kSizeCString);
        // set UUID
        CFDictionarySetCString(properties, kHIDDeviceProperty_UUID, device.UUID, kSizeCString);
        // set descriptor
        CFDictionarySetData(properties, kHIDDeviceProperty_ReportDescriptor, device.HIDDescriptor,
                device.HIDDescriptorLen);
        // set country code
        uint8_t countryCode = device.HIDCountryCode;
        CFDictionarySetNumber(properties, kHIDDeviceProperty_CountryCode, kCFNumberSInt8Type,
                &countryCode);
        // set vendor id
        uint16_t vid = device.HIDVendorId;
        CFDictionarySetNumber(properties, kHIDDeviceProperty_VendorID, kCFNumberSInt16Type, &vid);
        // set product id
        uint16_t pid = device.HIDProductId;
        CFDictionarySetNumber(properties, kHIDDeviceProperty_ProductID, kCFNumberSInt16Type, &pid);
        // set display UUID
        CFDictionarySetCString(properties, kHIDDeviceProperty_DisplayUUID, device.DisplayUUID,
                kSizeCString);

        HIDDeviceRef deviceRef = nullptr;
        OSStatus status = HIDDeviceCreateVirtual(&deviceRef, properties);

        // clean-up
        CFRelease(properties);

        if (status != kNoErr)
        {
            LOG_ERROR((dipo, "failed to attach HID device (HIDDeviceCreateVirtual)"));
            return false;
        }

        if (kNoErr != (status = HIDRegisterDevice(deviceRef)))
        {
            LOG_ERROR((dipo, "failed to attach HID device (HIDRegisterDevice)"));
            return false;
        }

        // successful, store device
        hidDevices.insert(std::pair<string, HIDDeviceRef>(device.UUID, deviceRef));
    }
    else
    {
        LOG_WARN((dipo, "failed to attach HID device, used uuid %s is already registered",
                device.UUID));
        // still return true
    }

    return true;
}

bool InputChannel::DetachInput(const std::string& uuid)
{
    OSStatus status;
    auto device = hidDevices.find(uuid);

    if (device != hidDevices.end())
    {
        LOGD_DEBUG((dipo, "HID detach: %s", uuid.c_str()));
        auto ref = device->second;

        if (kNoErr != (status = HIDDeregisterDevice(ref)))
        {
            LOG_ERROR((dipo, "failed to detach HID device with UUID %s", uuid.c_str()));
            return false;
        }

        HIDDeviceForget(&ref);

        hidDevices.erase(uuid);
    }
    else
    {
        /* Device is already removed from InputChannel destructor */
    }

    return true;
}

bool InputChannel::SendInput(const HIDInputReport& report)
{
    if (hidDevices.end() == hidDevices.find(report.UUID))
    {
        LOG_WARN((dipo, "ignore input, device with uuid %s is not registered", report.UUID));
        return false;
    }

    // verbose logging
    if (verbose)
    {
        const size_t len = report.HIDReportLen * 3 + 1;
        char buffer[len];
        for (unsigned int i = 0; i < report.HIDReportLen; i++)
            sprintf(buffer + i * 3, "%02x ", report.HIDReportData[i]);
        LOGD_VERBOSE((dipo, "HID input %s: %s", report.UUID, buffer));
    }

/*#if DIPO_DEBUG_MEASURE_VIDEO
    {
        uint64_t now = GetCurrentTimeNano();
        printf("PERF %s %.14llu\n", __PRETTY_FUNCTION__, now);
    }
#endif*/ // DIPO_DEBUG_MEASURE_VIDEO

    auto request = CFDictionaryCreateMutable(NULL, 0, &kCFTypeDictionaryKeyCallBacks,
            &kCFTypeDictionaryValueCallBacks);
    dipo_return_value_on_null(dipo, request, false);

    CFDictionarySetValue(request, CFSTR(kAirPlayKey_Type), CFSTR(kAirPlayCommand_HIDSendReport) );
    CFDictionarySetCString(request, CFSTR(kAirPlayKey_UUID), report.UUID, -1);
    CFDictionarySetData(request, CFSTR(kAirPlayKey_HIDReport), report.HIDReportData,
            report.HIDReportLen);

    // TODO check timestamp format
    CFDictionarySetInt64(request, CFSTR(kAirPlayKey_Timestamp), 0);

    OSStatus status = AirPlayReceiverSessionSendCommand(session.GetReference(), request, nullptr,
            nullptr);
    CFRelease(request);

    if (status != kNoErr)
    {
        LOG_ERROR((dipo, "failed to send HID input report (status = %d)", status));
        return false;
    }

    return true;
}

} } // namespace adit { namespace carplay
